import glob
import os
from pathlib import Path

import pandas as pd

# Script que varre as planilhas da pasta, extrai cabeçalho + itens de cada nota
# e consolida tudo em um único Excel normalizado.

# ============================================================
# CONFIGURACOES
# ============================================================

BASE_DIR = Path(__file__).resolve().parent
PASTA = BASE_DIR  # pasta onde estao as planilhas
ARQUIVO_SAIDA = BASE_DIR / "notas_normalizadas.xlsx"


# ============================================================
# FUNCOES AUXILIARES
# ============================================================

def is_vazio(valor):
    return valor is None or (isinstance(valor, str) and valor.strip() == "")


def is_header_item(linha):
    if not linha or len(linha) < 2:
        return False
    primeira = str(linha[0]).strip().lower()
    segunda = str(linha[1]).strip().lower()
    return primeira == "produto" and segunda == "descricao"


def is_controle_linha(linha):
    """True para linhas de controle (cond pagto, totais, etc.) que nao devem virar item."""
    if not linha:
        return False
    if not isinstance(linha[0], str):
        return False
    valor = linha[0].strip().lower()
    return valor.startswith("cond") or valor.startswith("tot") or valor == "documento"


# ============================================================
# FUNCAO PARA PROCESSAR UM UNICO ARQUIVO
# ============================================================

def processar_arquivo(caminho_arquivo: os.PathLike) -> list[dict]:
    df_original = pd.read_excel(caminho_arquivo, header=None)
    # Substitui NaN por None para facilitar os testes de vazio
    linhas = df_original.where(pd.notnull(df_original), None).values.tolist()

    doc_atual = {}
    itens = []

    for i, linha in enumerate(linhas):
        # -------------------------------------------------------
        # 1) DETECTA O INICIO DE UM DOCUMENTO
        # -------------------------------------------------------
        if isinstance(linha[0], str) and linha[0].strip().lower() == "documento":
            if i + 1 < len(linhas):
                info = linhas[i + 1]
                doc_atual = {
                    "Documento": info[0],
                    "Serie": info[1],
                    "DT_Emissao": info[2],
                    "DT_Digitacao": info[3],
                    "Tipo_Docto": info[4],
                    "Forn_Cliente": info[5],
                    "Loja": info[6],
                    "Razao_Social": info[7],
                }

        # -------------------------------------------------------
        # 2) DETECTA INICIO DA LISTA DE ITENS
        # -------------------------------------------------------
        if isinstance(linha[0], str) and linha[0].strip().lower() == "produto":
            j = i + 1
            while j < len(linhas) and not is_vazio(linhas[j][0]):
                item_linha = linhas[j]
                if is_controle_linha(item_linha):
                    break  # chegou em cond pagto ou total, encerra itens
                if is_header_item(item_linha):
                    j += 1
                    continue
                item = {
                    "Documento": doc_atual.get("Documento"),
                    "Serie": doc_atual.get("Serie"),
                    "DT_Emissao": doc_atual.get("DT_Emissao"),
                    "DT_Digitacao": doc_atual.get("DT_Digitacao"),
                    "Tipo_Docto": doc_atual.get("Tipo_Docto"),
                    "Forn_Cliente": doc_atual.get("Forn_Cliente"),
                    "Loja": doc_atual.get("Loja"),
                    "Razao_Social": doc_atual.get("Razao_Social"),
                    "Produto": item_linha[0],
                    "Descricao": item_linha[1],
                    "Cd_Fisc": item_linha[2],
                    "TES": item_linha[3],
                    "ICMS": item_linha[4],
                    "IPI": item_linha[5],
                    "Armazem": item_linha[6],
                    "Quantidade": item_linha[7],
                    "Vlr_Unitario": item_linha[8],
                    "Vlr_Mercad": item_linha[9],
                }
                itens.append(item)
                j += 1

        # -------------------------------------------------------
        # 3) DETECTA CONDICAO DE PAGAMENTO
        # -------------------------------------------------------
        if isinstance(linha[0], str) and "cond" in linha[0].strip().lower():
            # propaga condicao de pagamento para todos os itens do documento atual
            cond_pag = linhas[i + 1][0] if i + 1 < len(linhas) else None
            for item in reversed(itens):
                if item["Documento"] == doc_atual.get("Documento"):
                    item["Cond_Pagto"] = cond_pag
                else:
                    break

    return itens


# ============================================================
# PROCESSA TODOS OS ARQUIVOS DA PASTA
# ============================================================

todos_itens = []

for arquivo in glob.glob(os.path.join(PASTA, "*.xlsx")):
    nome = Path(arquivo).name
    if nome.startswith("~$"):
        continue  # ignora arquivos temporarios do Excel
    if Path(arquivo).resolve() == ARQUIVO_SAIDA:
        continue
    print(f"Processando: {arquivo}")
    itens = processar_arquivo(arquivo)
    todos_itens.extend(itens)

# ============================================================
# SALVANDO ARQUIVO FINAL
# ============================================================

if todos_itens:
    df_final = pd.DataFrame(todos_itens)
    df_final.to_excel(ARQUIVO_SAIDA, index=False)
    print("Concluido! Arquivo salvo em:", ARQUIVO_SAIDA)
else:
    print("Nenhum item encontrado.")
